# coding: utf-8
from odoo.addons.account.tests.common import AccountTestInvoicingCommon
from odoo.tests import tagged

import base64
from freezegun import freeze_time

@tagged('post_install_l10n', "post_install", "-at_install")
class Test1099(AccountTestInvoicingCommon):
    @classmethod
    def setUpClass(cls, chart_template_ref="l10n_generic_coa.configurable_chart_template"):
        super().setUpClass(chart_template_ref=chart_template_ref)

        cls.env.company.write({
            "street": "1 W Seneca St",
            "street2": "Floor 26",
            "city": "Buffalo",
            "state_id": cls.env["res.country.state"].search([("name", "=", "New York")], limit=1).id,
            "zip": "14203",
            "country_id": cls.env.ref("base.us").id,
            "phone": "(716) 249-2880",
            "vat": "987654321",
        })

        cls.vendor_non_1099 = cls.env["res.partner"].create({
            "name": "non-1099 vendor",
        })

        cls.vendor_1099 = cls.env["res.partner"].create({
            "name": "1099 vendor",
            "street": "250 Executive Park Blvd",
            "street2": "#3400",
            "city": "San Francisco",
            "state_id": cls.env["res.country.state"].search([("name", "=", "California")], limit=1).id,
            "zip": "94134",
            "country_id": cls.env.ref("base.us").id,
            "phone": "(650) 691-3277",
            "vat": "123456789",
            "email": "vendor@example.com",
            "box_1099_id": cls.env.ref("l10n_us_1099.box_1099_misc_03").id,
        })

        cls.liquidity_account = cls.env["account.account"].search(
            [("account_type", "=", "asset_cash")],
            limit=1
        )
        cls.expense_account = cls.env["account.account"].search(
            [("account_type", "=", "expense")],
            limit=1
        )

    @freeze_time("2021-02-10")
    def test1099Wizard(self):
        wizard = self.env["l10n_us_1099.wizard"].create({})
        wizard.lines_to_export = self.env["account.move.line"]

        move_vals = {
            "date": "2020-06-06",  # wizard includes last year by default
            "line_ids": [
                (0, 0, {
                    "name": "debit",
                    "partner_id": self.vendor_non_1099.id,
                    "account_id": self.expense_account.id,
                    "debit": 100.0,
                    "credit": 0.0,
                }),
                (0, 0, {
                    "name": "credit",
                    "partner_id": self.vendor_non_1099.id,
                    "account_id": self.liquidity_account.id,
                    "debit": 0.0,
                    "credit": 100.0,
                }),
            ],
        }
        move = self.env["account.move"].create(move_vals)
        move.action_post()

        move_vals["line_ids"][0][2]["partner_id"] = self.vendor_1099.id
        move_vals["line_ids"][1][2]["partner_id"] = self.vendor_1099.id
        move = self.env["account.move"].create(move_vals)
        move.action_post()
        expected_lines = move.line_ids.filtered("credit")

        move_vals["line_ids"][0][2]["debit"] = 150
        move_vals["line_ids"][1][2]["credit"] = 150
        move = self.env["account.move"].create(move_vals)
        move.action_post()
        expected_lines |= move.line_ids.filtered("credit")

        wizard._compute_lines_to_export()
        self.assertEqual(wizard.lines_to_export, expected_lines, "Wizard should contain the credit part of the 1099 vendor entry.")

        wizard.action_generate()
        csv_content = base64.b64decode(wizard.generated_csv_file).decode().splitlines()

        self.maxDiff = None  # show the full diff in case of errors
        header = (
            "Payer Name,Payer Address Line 1,Payer Address Line 2,Payer City,Payer State,Payer Zip,Payer Country,Payer Phone Number,Payer TIN,"
            "Payee Name,Payee Address Line 1,Payee Address Line 2,Payee City,Payee State,Payee Zip,Payee Country,Payee Email,Payee TIN,NEC - 1 Nonemployee compensation,"
            "MISC - 1 Rents,MISC - 2 Royalties,MISC - 3 Other income,MISC - 5 Fishing boat proceeds,MISC - 6 Medical and health care payments,"
            "MISC - 8 Substitute payments in lieu of dividends or interest,MISC - 9 Crop insurance proceeds,MISC - 10 Gross proceeds paid to an attorney,"
            "MISC - 11 Fish purchased for resale,MISC - 12 Section 409A deferrals,MISC - 13 Excess golden parachute payments,MISC - 14 Nonqualified deferred compensation"
        )
        self.assertEqual(csv_content[0], header, "Wizard did not generate the expected header.")

        expected_line = (
            "company_1_data,1 W Seneca St,Floor 26,Buffalo,New York,14203,United States,(716) 249-2880,987654321,1099 vendor,250 Executive Park Blvd,"
            "#3400,San Francisco,California,94134,United States,vendor@example.com,123456789,0,0,0,250.0,0,0,0,0,0,0,0,0,0"
        )
        self.assertEqual(csv_content[1], expected_line, "Wizard did not generate the expected line for the 1099 vendor.")
        self.assertEqual(len(csv_content), 2, "Wizard should exactly generate the two lines above.")
