# -*- coding: utf-8 -*-
# Part of Odoo. See LICENSE file for full copyright and licensing details.
from contextlib import contextmanager
from unittest.mock import Mock, patch

from odoo.tests.common import TransactionCase, tagged, Form


@tagged('-standard', 'external')
class TestDeliveryUPS(TransactionCase):

    def setUp(self):
        super(TestDeliveryUPS, self).setUp()

        self.iPadMini = self.env.ref('product.product_product_6')
        self.large_desk = self.env.ref('product.product_product_8')
        self.uom_unit = self.env.ref('uom.product_uom_unit')

        # Add a full address to "Your Company" and "Agrolait"
        self.your_company = self.env.ref('base.main_partner')
        self.your_company.write({'country_id': self.env.ref('base.us').id,
                                 'state_id': self.env.ref('base.state_us_5').id,
                                 'city': 'San Francisco',
                                 'street': '51 Federal Street',
                                 'zip': '94107'})
        self.agrolait = self.env.ref('base.res_partner_2')
        self.agrolait.write({'country_id': self.env.ref('base.be').id,
                             'city': 'Auderghem-Ouderghem',
                             'street': 'Avenue Edmond Van Nieuwenhuyse',
                             'zip': '1160'})
        self.stock_location = self.env.ref('stock.stock_location_stock')
        self.customer_location = self.env.ref('stock.stock_location_customers')

    def wiz_put_in_pack(self, picking):
        """ Helper to use the 'choose.delivery.package' wizard
        in order to call the 'action_put_in_pack' method.
        """
        wiz_action = picking.action_put_in_pack()
        self.assertEqual(wiz_action['res_model'], 'choose.delivery.package', 'Wrong wizard returned')
        wiz = Form(self.env[wiz_action['res_model']].with_context(wiz_action['context']).create({
            'delivery_package_type_id': picking.carrier_id.ups_default_package_type_id.id
        }))
        choose_delivery_carrier = wiz.save()
        choose_delivery_carrier.action_put_in_pack()

    def test_01_ups_basic_flow(self):
        SaleOrder = self.env['sale.order']

        sol_vals = {'product_id': self.iPadMini.id,
                    'name': "[A1232] Large Cabinet",
                    'product_uom': self.uom_unit.id,
                    'product_uom_qty': 1.0,
                    'price_unit': self.iPadMini.lst_price}

        # Set service type = 'UPS Worldwide Expedited', which is available between US to BE
        carrier = self.env.ref('delivery_ups.delivery_carrier_ups_us')
        carrier.write({'ups_default_service_type': '08',
                       'ups_package_dimension_unit': 'IN'})
        carrier.ups_default_package_type_id.write({'height': '3',
                                                   'width': '3',
                                                   'packaging_length': '3'})

        so_vals = {'partner_id': self.agrolait.id,
                   'order_line': [(0, None, sol_vals)]}

        sale_order = SaleOrder.create(so_vals)
        # I add delivery cost in Sales order
        delivery_wizard = Form(self.env['choose.delivery.carrier'].with_context({
            'default_order_id': sale_order.id,
            'default_carrier_id': carrier.id
        }))
        choose_delivery_carrier = delivery_wizard.save()
        choose_delivery_carrier.update_price()
        self.assertGreater(choose_delivery_carrier.delivery_price, 0.0, "UPS delivery cost for this SO has not been correctly estimated.")
        choose_delivery_carrier.button_confirm()

        sale_order.action_confirm()
        self.assertEqual(len(sale_order.picking_ids), 1, "The Sales Order did not generate a picking.")

        picking = sale_order.picking_ids[0]
        self.assertEqual(picking.carrier_id.id, sale_order.carrier_id.id, "Carrier is not the same on Picking and on SO.")

        picking.move_ids[0].quantity_done = 1.0
        self.assertGreater(picking.shipping_weight, 0.0, "Picking weight should be positive.")

        picking._action_done()
        self.assertIsNot(picking.carrier_tracking_ref, False, "UPS did not return any tracking number")
        self.assertGreater(picking.carrier_price, 0.0, "UPS carrying price is probably incorrect")

        picking.cancel_shipment()
        self.assertFalse(picking.carrier_tracking_ref, "Carrier Tracking code has not been properly deleted")
        self.assertEqual(picking.carrier_price, 0.0, "Carrier price has not been properly deleted")

    def test_02_ups_multipackage_flow(self):
        SaleOrder = self.env['sale.order']

        # Set package type = 'Pallet' and service type = 'UPS Worldwide Express Freight'
        # so in this case height, width and length required.
        carrier = self.env.ref('delivery_ups.delivery_carrier_ups_us')
        carrier.write({'ups_default_package_type_id': self.env.ref('delivery_ups.ups_packaging_30').id,
                       'ups_default_service_type': '96',
                       'ups_package_dimension_unit': 'IN'})
        carrier.ups_default_package_type_id.write({'height': '3',
                                                   'width': '3',
                                                   'packaging_length': '3'})

        sol_1_vals = {'product_id': self.iPadMini.id,
                      'name': "[A1232] Large Cabinet",
                      'product_uom': self.uom_unit.id,
                      'product_uom_qty': 1.0,
                      'price_unit': self.iPadMini.lst_price}

        sol_2_vals = {'product_id': self.large_desk.id,
                      'name': "[A1090] Large Desk",
                      'product_uom': self.uom_unit.id,
                      'product_uom_qty': 1.0,
                      'price_unit': self.large_desk.lst_price}

        so_vals = {'partner_id': self.agrolait.id,
                   'order_line': [(0, None, sol_1_vals), (0, None, sol_2_vals)]}

        sale_order = SaleOrder.create(so_vals)
        # I add delivery cost in Sales order
        delivery_wizard = Form(self.env['choose.delivery.carrier'].with_context({
            'default_order_id': sale_order.id,
            'default_carrier_id': carrier.id
        }))
        choose_delivery_carrier = delivery_wizard.save()
        choose_delivery_carrier.update_price()
        self.assertGreater(choose_delivery_carrier.delivery_price, 0.0, "UPS delivery cost for this SO has not been correctly estimated.")
        choose_delivery_carrier.button_confirm()

        sale_order.action_confirm()
        self.assertEqual(len(sale_order.picking_ids), 1, "The Sales Order did not generate a picking.")

        picking = sale_order.picking_ids[0]
        self.assertEqual(picking.carrier_id.id, sale_order.carrier_id.id, "Carrier is not the same on Picking and on SO.")

        move0 = picking.move_ids[0]
        move0.quantity_done = 1.0
        self.wiz_put_in_pack(picking)
        move1 = picking.move_ids[1]
        move1.quantity_done = 1.0
        self.wiz_put_in_pack(picking)
        self.assertEqual(len(picking.move_line_ids.mapped('result_package_id')), 2, "2 packages should have been created at this point")
        self.assertGreater(picking.shipping_weight, 0.0, "Picking weight should be positive.")

        picking._action_done()
        self.assertIsNot(picking.carrier_tracking_ref, False, "UPS did not return any tracking number")
        self.assertGreater(picking.carrier_price, 0.0, "UPS carrying price is probably incorrect")

        picking.cancel_shipment()
        self.assertFalse(picking.carrier_tracking_ref, "Carrier Tracking code has not been properly deleted")
        self.assertEqual(picking.carrier_price, 0.0, "Carrier price has not been properly deleted")

    def test_03_ups_flow_from_delivery_order(self):
        # Set service type = 'UPS Worldwide Expedited', which is available between US to BE
        carrier = self.env.ref('delivery_ups.delivery_carrier_ups_us')
        carrier.write({'ups_default_service_type': '08',
                       'ups_package_dimension_unit': 'IN'})
        carrier.ups_default_package_type_id.write({'height': '3',
                                                   'width': '3',
                                                   'packaging_length': '3'})

        StockPicking = self.env['stock.picking']

        order1_vals = {
                    'product_id': self.iPadMini.id,
                    'name': "[A1232] iPad Mini",
                    'product_uom': self.uom_unit.id,
                    'product_uom_qty': 1.0,
                    'location_id': self.stock_location.id,
                    'location_dest_id': self.customer_location.id}

        do_vals = { 'partner_id': self.agrolait.id,
                    'carrier_id': carrier.id,
                    'location_id': self.stock_location.id,
                    'location_dest_id': self.customer_location.id,
                    'picking_type_id': self.env.ref('stock.picking_type_out').id,
                    'move_ids_without_package': [(0, None, order1_vals)]}

        delivery_order = StockPicking.create(do_vals)
        self.assertEqual(delivery_order.state, 'draft', 'Shipment state should be draft.')

        delivery_order.action_confirm()
        self.assertEqual(delivery_order.state, 'assigned', 'Shipment state should be ready(assigned).')
        delivery_order.move_ids_without_package.quantity_done = 1.0

        delivery_order.button_validate()
        self.assertEqual(delivery_order.state, 'done', 'Shipment state should be done.')

    def test_04_backorder_and_track_number(self):
        """ Suppose a two-steps delivery with 2 x Product A and 2 x Product B.
        For the Pick step, process a first picking (PICK01) with 2 x Product A
        and a backorder (PICK02) with 2 x Product B
        For the Out step, process a first picking (OUT01) with 1 x Product A
        and a backorder (OUT02) with 1 x Product A and 2 x Product B
        This test ensures that:
            - OUT01 and OUT02 have their own tracking reference
            - The tracking reference of PICK01 is defined with the one of OUT01 and OUT02
            - The tracking reference of PICK02 is defined with the one of OUT02
        """
        def process_picking(picking):
            action = picking.button_validate()
            wizard = Form(self.env[action['res_model']].with_context(action['context']))
            wizard.save().process()

        warehouse = self.env.user._get_default_warehouse_id()
        warehouse.delivery_steps = 'pick_ship'
        stock_location = warehouse.lot_stock_id

        carrier = self.env.ref('delivery_ups.delivery_carrier_ups_us')
        carrier.write({'ups_default_service_type': '08', 'ups_package_dimension_unit': 'IN'})
        carrier.ups_default_package_type_id.write({'height': '1', 'width': '1', 'packaging_length': '1'})

        product_a, product_b = self.env['product.product'].create([{
            'name': p_name,
            'weight': 1,
        } for p_name in ['Product A', 'Product B']])

        so_form = Form(self.env['sale.order'])
        so_form.partner_id = self.agrolait
        with so_form.order_line.new() as line:
            line.product_id = product_a
            line.product_uom_qty = 2
        with so_form.order_line.new() as line:
            line.product_id = product_b
            line.product_uom_qty = 2
        so = so_form.save()

        # Add UPS shipping
        delivery_wizard = Form(self.env['choose.delivery.carrier'].with_context({
            'default_order_id': so.id,
            'default_carrier_id': carrier.id
        }))
        choose_delivery_carrier = delivery_wizard.save()
        choose_delivery_carrier.update_price()
        choose_delivery_carrier.button_confirm()

        so.action_confirm()
        pick01 = so.picking_ids.filtered(lambda p: p.location_id == stock_location)
        out01 = so.picking_ids - pick01

        # First step with 2 x Product A
        pick01.move_ids.filtered(lambda m: m.product_id == product_a).quantity_done = 2
        process_picking(pick01)
        # First step with 2 x Product B
        pick02 = pick01.backorder_ids
        process_picking(pick02)

        # Second step with 1 x Product A
        out01.move_ids.filtered(lambda m: m.product_id == product_a).quantity_done = 1
        process_picking(out01)
        out02 = out01.backorder_ids
        self.assertTrue(out01.carrier_tracking_ref)
        self.assertFalse(out02.carrier_tracking_ref)
        self.assertEqual(pick01.carrier_tracking_ref, out01.carrier_tracking_ref)
        self.assertFalse(pick02.carrier_tracking_ref)

        # Second step with 1 x Product A + 2 x Product B
        process_picking(out02)
        self.assertTrue(out01.carrier_tracking_ref)
        self.assertTrue(out02.carrier_tracking_ref)
        self.assertEqual(pick01.carrier_tracking_ref, out01.carrier_tracking_ref + ',' + out02.carrier_tracking_ref)
        self.assertEqual(pick02.carrier_tracking_ref, out02.carrier_tracking_ref)


@tagged('standard', '-external')
class TestMockDeliveryUPS(TestDeliveryUPS):

    @contextmanager
    def patch_ups_requests(self):
        """ Mock context for requests to the UPS API. """

        class MockedSession:
            def __init__(self, *args, **kwargs):
                self.headers = dict()

            def mount(self, *args, **kwargs):
                return None

            def close(self, *args, **kwargs):
                return None

            def post(self, *args, **kwargs):
                response = Mock()
                response.headers = {}
                response.status_code = 200
                if b'<ns0:RateRequest' in kwargs.get('data'):
                    response.content = b'<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"><soapenv:Header/><soapenv:Body><rate:RateResponse xmlns:rate="http://www.ups.com/XMLSchema/XOLTWS/Rate/v1.1"><common:Response xmlns:common="http://www.ups.com/XMLSchema/XOLTWS/Common/v1.0"><common:ResponseStatus><common:Code>1</common:Code><common:Description>Success</common:Description></common:ResponseStatus><common:Alert><common:Code>110971</common:Code><common:Description>Your invoice may vary from the displayed reference rates</common:Description></common:Alert><common:Alert><common:Code>120900</common:Code><common:Description>User Id and Shipper Number combination is not qualified to receive negotiated rates</common:Description></common:Alert><common:TransactionReference/></common:Response><rate:RatedShipment><rate:Service><rate:Code>08</rate:Code><rate:Description/></rate:Service><rate:RatedShipmentAlert><rate:Code>120900</rate:Code><rate:Description>User Id and Shipper Number combination is not qualified to receive negotiated rates.</rate:Description></rate:RatedShipmentAlert><rate:RatedShipmentAlert><rate:Code>110971</rate:Code><rate:Description>Your invoice may vary from the displayed reference rates</rate:Description></rate:RatedShipmentAlert><rate:BillingWeight><rate:UnitOfMeasurement><rate:Code>KGS</rate:Code><rate:Description>Kilograms</rate:Description></rate:UnitOfMeasurement><rate:Weight>4.0</rate:Weight></rate:BillingWeight><rate:TransportationCharges><rate:CurrencyCode>EUR</rate:CurrencyCode><rate:MonetaryValue>35.43</rate:MonetaryValue></rate:TransportationCharges><rate:ServiceOptionsCharges><rate:CurrencyCode>EUR</rate:CurrencyCode><rate:MonetaryValue>0.00</rate:MonetaryValue></rate:ServiceOptionsCharges><rate:TotalCharges><rate:CurrencyCode>EUR</rate:CurrencyCode><rate:MonetaryValue>35.43</rate:MonetaryValue></rate:TotalCharges><rate:RatedPackage><rate:Weight>4.0</rate:Weight></rate:RatedPackage></rate:RatedShipment></rate:RateResponse></soapenv:Body></soapenv:Envelope>'
                elif b'<ns0:ShipmentRequest' in kwargs.get('data'):
                    response.content = b'<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"><soapenv:Header/><soapenv:Body><ship:ShipmentResponse xmlns:ship="http://www.ups.com/XMLSchema/XOLTWS/Ship/v1.0"><common:Response xmlns:common="http://www.ups.com/XMLSchema/XOLTWS/Common/v1.0"><common:ResponseStatus><common:Code>1</common:Code><common:Description>Success</common:Description></common:ResponseStatus><common:Alert><common:Code>120900</common:Code><common:Description>User Id and Shipper Number combination is not qualified to receive negotiated rates</common:Description></common:Alert><common:TransactionReference/></common:Response><ship:ShipmentResults><ship:ShipmentCharges><ship:TransportationCharges><ship:CurrencyCode>EUR</ship:CurrencyCode><ship:MonetaryValue>32.35</ship:MonetaryValue></ship:TransportationCharges><ship:ServiceOptionsCharges><ship:CurrencyCode>EUR</ship:CurrencyCode><ship:MonetaryValue>0.00</ship:MonetaryValue></ship:ServiceOptionsCharges><ship:TotalCharges><ship:CurrencyCode>EUR</ship:CurrencyCode><ship:MonetaryValue>32.35</ship:MonetaryValue></ship:TotalCharges></ship:ShipmentCharges><ship:BillingWeight><ship:UnitOfMeasurement><ship:Code>KGS</ship:Code><ship:Description>Kilograms</ship:Description></ship:UnitOfMeasurement><ship:Weight>1.0</ship:Weight></ship:BillingWeight><ship:ShipmentIdentificationNumber>1ZXXXXXXXXXXXXXXXX</ship:ShipmentIdentificationNumber><ship:PackageResults><ship:TrackingNumber>1ZXXXXXXXXXXXXXXXX</ship:TrackingNumber><ship:ServiceOptionsCharges><ship:CurrencyCode>EUR</ship:CurrencyCode><ship:MonetaryValue>0.00</ship:MonetaryValue></ship:ServiceOptionsCharges><ship:ShippingLabel><ship:ImageFormat><ship:Code>GIF</ship:Code><ship:Description>GIF</ship:Description></ship:ImageFormat><ship:GraphicImage>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</ship:GraphicImage><ship:HTMLImage>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</ship:HTMLImage></ship:ShippingLabel></ship:PackageResults><ship:Form><ship:Code>01</ship:Code><ship:Description>All Requested International Forms</ship:Description><ship:Image><ship:ImageFormat><ship:Code>PDF</ship:Code><ship:Description>PDF</ship:Description></ship:ImageFormat><ship:GraphicImage>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</ship:GraphicImage></ship:Image></ship:Form></ship:ShipmentResults></ship:ShipmentResponse></soapenv:Body></soapenv:Envelope>'
                elif b'<ns0:VoidShipmentRequest' in kwargs.get('data'):
                    response.content = b'<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"><soapenv:Header/><soapenv:Body><v1:VoidShipmentResponse xmlns:v1="http://www.ups.com/XMLSchema/XOLTWS/Void/v1.1"><v11:Response xmlns:v11="http://www.ups.com/XMLSchema/XOLTWS/Common/v1.0"><v11:ResponseStatus><v11:Code>1</v11:Code><v11:Description>Success</v11:Description></v11:ResponseStatus><v11:TransactionReference><v11:CustomerContext>Cancle shipment</v11:CustomerContext><v11:TransactionIdentifier>sxlhcXhZ7gc40ty6Rq</v11:TransactionIdentifier></v11:TransactionReference></v11:Response><v1:SummaryResult><v1:Status><v1:Code>1</v1:Code><v1:Description>Voided</v1:Description></v1:Status></v1:SummaryResult></v1:VoidShipmentResponse></soapenv:Body></soapenv:Envelope>'
                return response

        # zeep.Client.transport is using post from requests.Session
        with patch('zeep.transports.requests.Session') as mocked_session:
            mocked_session.side_effect = MockedSession
            yield mocked_session

    def test_01_ups_basic_flow(self):
        with self.patch_ups_requests():
            super().test_01_ups_basic_flow()

    def test_02_ups_multipackage_flow(self):
        with self.patch_ups_requests():
            super().test_02_ups_multipackage_flow()

    def test_03_ups_flow_from_delivery_order(self):
        with self.patch_ups_requests():
            super().test_03_ups_flow_from_delivery_order()

    def test_04_backorder_and_track_number(self):
        with self.patch_ups_requests():
            super().test_04_backorder_and_track_number()
